/*
 * SPDX-License-Identifier: LGPL-2.1-or-later
 *
 * Copyright (C) 2018 Red Hat, Inc.
 */

import React from 'react';
import cockpit from 'cockpit';
import { InputGroup } from "@patternfly/react-core/dist/esm/components/InputGroup";
import { FormGroup } from "@patternfly/react-core/dist/esm/components/Form";
import { FormSelect, FormSelectOption } from "@patternfly/react-core/dist/esm/components/FormSelect";
import { Grid, GridItem } from "@patternfly/react-core/dist/esm/layouts/Grid";
import { Slider } from "@patternfly/react-core/dist/esm/components/Slider";
import { TextInput } from "@patternfly/react-core/dist/esm/components/TextInput";

import { FormHelper } from 'cockpit-components-form-helper.jsx';

import { digitFilter, units } from "../../../helpers.js";

import './memorySelectRow.css';

const _ = cockpit.gettext;

interface MemorySelectRowProps {
    id: string;
    value: number;
    label: string;
    minValue: number,
    maxValue: number,
    initialUnit: string,
    onValueChange: (val: number) => void,
    onUnitChange: (event: React.FormEvent, val: string) => void,
    isDisabled?: boolean;
    helperText?: string | null | undefined;
}

interface MemorySelectRowState {
    memory: string;
    inputHasFocus?: boolean;
}

export class MemorySelectRow extends React.Component<MemorySelectRowProps, MemorySelectRowState> {
    constructor(props: MemorySelectRowProps) {
        super(props);
        this.state = { memory: props.value.toFixed(0) };
    }

    static getDerivedStateFromProps(nextProps: MemorySelectRowProps, prevState: MemorySelectRowState) {
        if (!prevState.inputHasFocus)
            return { memory: nextProps.value.toFixed(0) };
        return null;
    }

    render() {
        const { id, value, label, minValue, maxValue, initialUnit, onValueChange, onUnitChange, isDisabled, helperText } = this.props;

        return (
            <FormGroup fieldId={id}
                       label={label}>
                <Grid hasGutter>
                    <GridItem span={12} sm={8}>
                        <Slider className='memory-slider'
                                id={id + '-slider'}
                                isDisabled={!!isDisabled}
                                key={id + '-slider-max-' + maxValue}
                                max={maxValue}
                                min={minValue}
                                onChange={(_, value) => onValueChange(value)}
                                showBoundaries
                                showTicks={false}
                                step={1}
                                value={value} />
                    </GridItem>
                    <GridItem span={12} sm={4}>
                        <InputGroup>
                            <TextInput id={id}
                                       className="ct-machines-size-input"
                                       type="text"
                                       inputMode="numeric"
                                       pattern="[0-9]*"
                                       value={this.state.memory}
                                       onKeyUp={digitFilter}
                                       isDisabled={!!isDisabled}
                                       onFocus={ () => this.setState({ inputHasFocus: true }) }
                                       onBlur={() => { onValueChange(Math.min(Math.max(minValue, Number(this.state.memory)), maxValue)); this.setState({ inputHasFocus: false }) }}
                                       onChange={(_, memory) => this.setState({ memory })} />
                            <FormSelect id={id + "-unit-select"}
                                        className="ct-machines-select-unit"
                                        value={initialUnit}
                                        isDisabled={isDisabled}
                                        onChange={onUnitChange}>
                                <FormSelectOption value={units.MiB.name} key={units.MiB.name}
                                                  label={_("MiB")} />
                                <FormSelectOption value={units.GiB.name} key={units.GiB.name}
                                                  label={_("GiB")} />
                            </FormSelect>
                        </InputGroup>
                    </GridItem>
                </Grid>
                <FormHelper helperText={helperText} />
            </FormGroup>
        );
    }
}
