/* -*- Mode: C++; tab-width: 2; indent-tabs-mode:nil; c-basic-offset: 2 -*- */
/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

#include "MozIntlHelper.h"
#include "nsBidiUtils.h"
#include "nsJSUtils.h"
#include "jsapi.h"
#include "js/experimental/Intl.h"   // JS::AddMozDateTimeFormatConstructor
#include "js/PropertyAndElement.h"  // JS_DefineFunctions
#include "js/PropertySpec.h"
#include "js/Wrapper.h"

using namespace mozilla;

NS_IMPL_ISUPPORTS(MozIntlHelper, mozIMozIntlHelper)

MozIntlHelper::MozIntlHelper() = default;

MozIntlHelper::~MozIntlHelper() = default;

static nsresult AddFunctions(JSContext* cx, JS::Handle<JS::Value> val,
                             const JSFunctionSpec* funcs) {
  if (!val.isObject()) {
    return NS_ERROR_INVALID_ARG;
  }

  // We might be adding functions to a Window.
  JS::Rooted<JSObject*> realIntlObj(
      cx, js::CheckedUnwrapDynamic(&val.toObject(), cx));
  if (!realIntlObj) {
    return NS_ERROR_INVALID_ARG;
  }

  JSAutoRealm ar(cx, realIntlObj);

  if (!JS_DefineFunctions(cx, realIntlObj, funcs)) {
    return NS_ERROR_FAILURE;
  }

  return NS_OK;
}

NS_IMETHODIMP
MozIntlHelper::AddGetCalendarInfo(JS::Handle<JS::Value> val, JSContext* cx) {
  static const JSFunctionSpec funcs[] = {
      JS_SELF_HOSTED_FN("getCalendarInfo", "Intl_getCalendarInfo", 1, 0),
      JS_FS_END};

  return AddFunctions(cx, val, funcs);
}

NS_IMETHODIMP
MozIntlHelper::AddDateTimeFormatConstructor(JS::Handle<JS::Value> val,
                                            JSContext* cx) {
  if (!val.isObject()) {
    return NS_ERROR_INVALID_ARG;
  }

  // We might be adding this constructor to a Window
  JS::Rooted<JSObject*> realIntlObj(
      cx, js::CheckedUnwrapDynamic(&val.toObject(), cx));
  if (!realIntlObj) {
    return NS_ERROR_INVALID_ARG;
  }

  JSAutoRealm ar(cx, realIntlObj);

  if (!JS::AddMozDateTimeFormatConstructor(cx, realIntlObj)) {
    return NS_ERROR_FAILURE;
  }

  return NS_OK;
}

NS_IMETHODIMP
MozIntlHelper::AddDisplayNamesConstructor(JS::Handle<JS::Value> val,
                                          JSContext* cx) {
  if (!val.isObject()) {
    return NS_ERROR_INVALID_ARG;
  }

  // We might be adding this constructor to a Window
  JS::Rooted<JSObject*> realIntlObj(
      cx, js::CheckedUnwrapDynamic(&val.toObject(), cx));
  if (!realIntlObj) {
    return NS_ERROR_INVALID_ARG;
  }

  JSAutoRealm ar(cx, realIntlObj);

  if (!JS::AddMozDisplayNamesConstructor(cx, realIntlObj)) {
    return NS_ERROR_FAILURE;
  }

  return NS_OK;
}

NS_IMETHODIMP
MozIntlHelper::StringHasRTLChars(JS::Handle<JS::Value> str, JSContext* cx,
                                 bool* res) {
  if (!str.isString()) {
    return NS_ERROR_INVALID_ARG;
  }

  nsAutoJSString string;
  if (!string.init(cx, str)) {
    return NS_ERROR_FAILURE;
  }

  *res = HasRTLChars(
      Span(static_cast<const char16_t*>(string.get()), string.Length()));
  return NS_OK;
}
