/**
 * Author......: See docs/credits.txt
 * License.....: MIT
 */

#include "common.h"
#include "types.h"
#include "modules.h"
#include "bitops.h"
#include "convert.h"
#include "shared.h"
#include "memory.h"

static const u32   ATTACK_EXEC    = ATTACK_EXEC_OUTSIDE_KERNEL;
static const u32   DGST_POS0      = 0;
static const u32   DGST_POS1      = 1;
static const u32   DGST_POS2      = 2;
static const u32   DGST_POS3      = 3;
static const u32   DGST_SIZE      = DGST_SIZE_4_16;
static const u32   HASH_CATEGORY  = HASH_CATEGORY_FDE;
static const char *HASH_NAME      = "LUKS v1 RIPEMD-160 + Twofish";
static const u64   KERN_TYPE      = 14643; // old kernel used here
static const u32   OPTI_TYPE      = OPTI_TYPE_ZERO_BYTE
                                  | OPTI_TYPE_SLOW_HASH_SIMD_LOOP;
static const u64   OPTS_TYPE      = OPTS_TYPE_STOCK_MODULE
                                  | OPTS_TYPE_PT_GENERATE_LE;
static const u32   SALT_TYPE      = SALT_TYPE_EMBEDDED;
static const char *ST_PASS        = "hashcat";
static const char *ST_HASH        = "$luks$1$ripemd160$twofish$cbc-plain64$128$107691$262bb8c34b794cb0f3fa2e093662c0461a2213a152d57fd442672cc724de2c31$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$bd3501cbb163b85622f2baa704177ad83b087002bbb43b841a88d25b24f87fe4c3a349f32f70a8338ef59aa09fdc10196285a0e5fc65fb273f5538dbcbc67ab28d40cc7855af9d4385d885640ee611d011ec5fea49c0dc57123caf845ee9887b0f4c721acfec5fbb788eb2fffb454e34d535c9ae2d3f27d62456b22649f2f978c8c5e00bcd9bb71cebba6d44f1a4f72f263c64045a49591fd272729dd52a7f28bb629cff0f6730485d6f5b7121074d668838596693a94400f8647ad97f06c99cb460bdb88eb0fb972300daecedd5ca1527a77fc9a3ccb282a0cd8baf51b182bd69518eecc25855d6c08e363ead0856afedf91c5c20c05d628b24e377c1a5c21861abb67674ecac7ffa090295aba303639016bdfcd09a0014a667d15079c980efc9b1aed8235cab7327da09d2cc3baeed2171224300e26e7c26acb05d726f618955fe3b1419ac2a1bf95b7d8e14edeade2ad5c3d48706f5ae615926f79aca797e57e42921734be8417d58c33220ce33c503c8f3dee0f4893b5365827da6a3ba07b85e24ea6f7a9141964c58e6743a711dc3b98124b6663f99cbf7c3690ec5100ed1b10e4a29f9f8124a78671018b4c135d17e86f3260f97ec3bb9acf58ab5a499c237a8e00e541f3ac0c22fb1256ae27edf2c4a69523da32508949159609717bf649fd4a52bd4f7814f5f864598aeedd13b36566a8b2e3ea346ac976dbf5fcbe6";

u32         module_attack_exec    (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return ATTACK_EXEC;     }
u32         module_dgst_pos0      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return DGST_POS0;       }
u32         module_dgst_pos1      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return DGST_POS1;       }
u32         module_dgst_pos2      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return DGST_POS2;       }
u32         module_dgst_pos3      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return DGST_POS3;       }
u32         module_dgst_size      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return DGST_SIZE;       }
u32         module_hash_category  (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return HASH_CATEGORY;   }
const char *module_hash_name      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return HASH_NAME;       }
u64         module_kern_type      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return KERN_TYPE;       }
u32         module_opti_type      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return OPTI_TYPE;       }
u64         module_opts_type      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return OPTS_TYPE;       }
u32         module_salt_type      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return SALT_TYPE;       }
const char *module_st_hash        (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return ST_HASH;         }
const char *module_st_pass        (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return ST_PASS;         }

#define LUKS_STRIPES        (                                   4000)
#define LUKS_SALT_LEN       (                                     32)
#define LUKS_SALT_HEX_LEN   (                      LUKS_SALT_LEN * 2)
#define LUKS_CT_LEN         (                                    512)
#define LUKS_CT_HEX_LEN     (                        LUKS_CT_LEN * 2)
#define LUKS_AF_MIN_LEN     (HC_LUKS_KEY_SIZE_128 / 8 * LUKS_STRIPES)
#define LUKS_AF_MIN_HEX_LEN (                    LUKS_AF_MIN_LEN * 2)
#define LUKS_AF_MAX_LEN     (HC_LUKS_KEY_SIZE_512 / 8 * LUKS_STRIPES)
#define LUKS_AF_MAX_HEX_LEN (                    LUKS_AF_MAX_LEN * 2)

typedef enum hc_luks_hash_type
{
  HC_LUKS_HASH_TYPE_SHA1      = 1,
  HC_LUKS_HASH_TYPE_SHA256    = 2,
  HC_LUKS_HASH_TYPE_SHA512    = 3,
  HC_LUKS_HASH_TYPE_RIPEMD160 = 4,
  HC_LUKS_HASH_TYPE_WHIRLPOOL = 5,

} hc_luks_hash_type_t;

typedef enum hc_luks_key_size
{
  HC_LUKS_KEY_SIZE_128 = 128,
  HC_LUKS_KEY_SIZE_256 = 256,
  HC_LUKS_KEY_SIZE_512 = 512,

} hc_luks_key_size_t;

typedef enum hc_luks_cipher_type
{
  HC_LUKS_CIPHER_TYPE_AES     = 1,
  HC_LUKS_CIPHER_TYPE_SERPENT = 2,
  HC_LUKS_CIPHER_TYPE_TWOFISH = 3,

} hc_luks_cipher_type_t;

typedef enum hc_luks_cipher_mode
{
  HC_LUKS_CIPHER_MODE_CBC_ESSIV_SHA256 = 1,
  HC_LUKS_CIPHER_MODE_CBC_PLAIN        = 2,
  HC_LUKS_CIPHER_MODE_CBC_PLAIN64      = 3,
  HC_LUKS_CIPHER_MODE_XTS_PLAIN        = 4,
  HC_LUKS_CIPHER_MODE_XTS_PLAIN64      = 5,

} hc_luks_cipher_mode_t;

typedef struct luks
{
  int hash_type;   // hc_luks_hash_type_t
  int key_size;    // hc_luks_key_size_t
  int cipher_type; // hc_luks_cipher_type_t
  int cipher_mode; // hc_luks_cipher_mode_t

  u32 ct_buf[LUKS_CT_LEN / 4];

  u32 af_buf[LUKS_AF_MAX_LEN / 4];
  u32 af_len;

} luks_t;

typedef struct luks_tmp
{
  u32 ipad32[8];
  u64 ipad64[8];

  u32 opad32[8];
  u64 opad64[8];

  u32 dgst32[32];
  u64 dgst64[16];

  u32 out32[32];
  u64 out64[16];

} luks_tmp_t;

static const char *SIGNATURE_LUKS = "$luks$1$ripemd160$twofish$";

bool module_unstable_warning (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra, MAYBE_UNUSED const hc_device_param_t *device_param)
{
  if ((device_param->opencl_platform_vendor_id == VENDOR_ID_APPLE) && (device_param->opencl_device_type & CL_DEVICE_TYPE_GPU))
  {
    if (device_param->is_metal == true)
    {
      if (strncmp (device_param->device_name, "Intel", 5) == 0)
      {
        // Intel Iris Graphics, Metal Version 244.303: failed to create 'm14643_init' pipeline, timeout reached
        return true;
      }
    }
  }

  return false;
}

u64 module_esalt_size (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra)
{
  const u64 esalt_size = (const u64) sizeof (luks_t);

  return esalt_size;
}

u64 module_tmp_size (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra)
{
  const u64 tmp_size = (const u64) sizeof (luks_tmp_t);

  return tmp_size;
}

int module_hash_decode (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED void *digest_buf, MAYBE_UNUSED salt_t *salt, MAYBE_UNUSED void *esalt_buf, MAYBE_UNUSED void *hook_salt_buf, MAYBE_UNUSED hashinfo_t *hash_info, const char *line_buf, MAYBE_UNUSED const int line_len)
{
  luks_t *luks = (luks_t *) esalt_buf;

  hc_token_t token;

  memset (&token, 0, sizeof (hc_token_t));

  token.token_cnt  = 7;

  token.signatures_cnt    = 1;
  token.signatures_buf[0] = SIGNATURE_LUKS;

  // signature with hash and cipher type
  token.len[0]     = 26;
  token.attr[0]    = TOKEN_ATTR_FIXED_LENGTH
                   | TOKEN_ATTR_VERIFY_SIGNATURE;

  // cipher mode
  token.sep[1]     = '$';
  token.len_min[1] = 9;
  token.len_max[1] = 16;
  token.attr[1]    = TOKEN_ATTR_VERIFY_LENGTH;

  // key size
  token.sep[2]     = '$';
  token.len[2]     = 3;
  token.attr[2]    = TOKEN_ATTR_FIXED_LENGTH
                   | TOKEN_ATTR_VERIFY_DIGIT;

  // iter
  token.sep[3]     = '$';
  token.len_min[3] = 1;
  token.len_max[3] = 10;
  token.attr[3]    = TOKEN_ATTR_VERIFY_LENGTH
                   | TOKEN_ATTR_VERIFY_DIGIT;

  // salt
  token.sep[4]     = '$';
  token.len[4]     = LUKS_SALT_HEX_LEN;
  token.attr[4]    = TOKEN_ATTR_FIXED_LENGTH
                   | TOKEN_ATTR_VERIFY_HEX;

  // af
  token.sep[5]     = '$';
  token.len_min[5] = LUKS_AF_MIN_HEX_LEN;
  token.len_max[5] = LUKS_AF_MAX_HEX_LEN;
  token.attr[5]    = TOKEN_ATTR_VERIFY_LENGTH
                   | TOKEN_ATTR_VERIFY_HEX;

  // ct
  token.sep[6]     = '$';
  token.len[6]     = LUKS_CT_HEX_LEN;
  token.attr[6]    = TOKEN_ATTR_FIXED_LENGTH
                   | TOKEN_ATTR_VERIFY_HEX;

  const int rc_tokenizer = input_tokenizer ((const u8 *) line_buf, line_len, &token);

  if (rc_tokenizer != PARSER_OK) return (rc_tokenizer);

  // hash type

  luks->hash_type = HC_LUKS_HASH_TYPE_RIPEMD160;

  // cipher type

  luks->cipher_type = HC_LUKS_CIPHER_TYPE_TWOFISH;

  // cipher mode

  const u8 *cipher_mode_pos = token.buf[1];
  const u32 cipher_mode_len = token.len[1];

  if ((strncmp ((const char *) cipher_mode_pos, "cbc-essiv:sha256", 16) == 0) && (cipher_mode_len == 16))
  {
    luks->cipher_mode = HC_LUKS_CIPHER_MODE_CBC_ESSIV_SHA256;
  }
  else if ((strncmp ((const char *) cipher_mode_pos, "cbc-plain",    9) == 0) && (cipher_mode_len ==  9))
  {
    luks->cipher_mode = HC_LUKS_CIPHER_MODE_CBC_PLAIN;
  }
  else if ((strncmp ((const char *) cipher_mode_pos, "cbc-plain64", 11) == 0) && (cipher_mode_len == 11))
  {
    luks->cipher_mode = HC_LUKS_CIPHER_MODE_CBC_PLAIN64;
  }
  else if ((strncmp ((const char *) cipher_mode_pos, "xts-plain",    9) == 0) && (cipher_mode_len ==  9))
  {
    luks->cipher_mode = HC_LUKS_CIPHER_MODE_XTS_PLAIN;
  }
  else if ((strncmp ((const char *) cipher_mode_pos, "xts-plain64", 11) == 0) && (cipher_mode_len == 11))
  {
    luks->cipher_mode = HC_LUKS_CIPHER_MODE_XTS_PLAIN64;
  }
  else
  {
    return (PARSER_LUKS_CIPHER_MODE);
  }

  // key size

  const u8 *key_size_pos = token.buf[2];

  const u32 key_size = hc_strtoul ((const char *) key_size_pos, NULL, 10);

  if (key_size == 128)
  {
    luks->key_size = HC_LUKS_KEY_SIZE_128;
  }
  else if (key_size == 256)
  {
    luks->key_size = HC_LUKS_KEY_SIZE_256;
  }
  else if (key_size == 512)
  {
    luks->key_size = HC_LUKS_KEY_SIZE_512;
  }
  else
  {
    return (PARSER_LUKS_KEY_SIZE);
  }

  // salt

  const u8 *salt_pos = token.buf[4];

  salt->salt_len = hex_decode (salt_pos, LUKS_SALT_HEX_LEN, (u8 *) salt->salt_buf);

  // iter

  const u8 *iter_pos = token.buf[3];

  const u32 iter = hc_strtoul ((const char *) iter_pos, NULL, 10);

  salt->salt_iter = iter - 1;

  // af

  u32 af_len = token.len[5] / 2;

  if (af_len != (key_size / 8) * LUKS_STRIPES) return (PARSER_LUKS_FILE_SIZE);

  const u8 *af_pos = token.buf[5];

  luks->af_len = hex_decode (af_pos, token.len[5], (u8 *) luks->af_buf);

  // ct

  u32 ct_len = token.len[6] / 2;

  if (ct_len != LUKS_CT_LEN) return (PARSER_LUKS_FILE_SIZE);

  const u8 *ct_pos = token.buf[6];

  hex_decode (ct_pos, LUKS_CT_HEX_LEN, (u8 *) luks->ct_buf);

  return (PARSER_OK);
}

int module_hash_encode (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const void *digest_buf, MAYBE_UNUSED const salt_t *salt, MAYBE_UNUSED const void *esalt_buf, MAYBE_UNUSED const void *hook_salt_buf, MAYBE_UNUSED const hashinfo_t *hash_info, char *line_buf, MAYBE_UNUSED const int line_size)
{
  const luks_t *luks = (const luks_t *) esalt_buf;

  // cipher mode

  const char *cipher_mode = "";

  switch (luks->cipher_mode)
  {
    case HC_LUKS_CIPHER_MODE_CBC_ESSIV_SHA256: cipher_mode = "cbc-essiv:sha256"; break;
    case HC_LUKS_CIPHER_MODE_CBC_PLAIN:        cipher_mode = "cbc-plain";        break;
    case HC_LUKS_CIPHER_MODE_CBC_PLAIN64:      cipher_mode = "cbc-plain64";      break;
    case HC_LUKS_CIPHER_MODE_XTS_PLAIN:        cipher_mode = "xts-plain";        break;
    case HC_LUKS_CIPHER_MODE_XTS_PLAIN64:      cipher_mode = "xts-plain64";      break;
    default:                                                                     return 0;
  }

  // key size

  u32 key_size = 0;

  switch (luks->key_size)
  {
    case HC_LUKS_KEY_SIZE_128: key_size = 128; break;
    case HC_LUKS_KEY_SIZE_256: key_size = 256; break;
    case HC_LUKS_KEY_SIZE_512: key_size = 512; break;
    default:                                   return 0;
  }

  // iterations

  u32 iterations = salt->salt_iter + 1;

  // salt

  char salt_buf[LUKS_SALT_HEX_LEN + 1] = { 0 };

  hex_encode ((const u8 *) salt->salt_buf, LUKS_SALT_LEN, (u8 *) salt_buf);

  // af

  char af_buf[LUKS_AF_MAX_HEX_LEN + 1] = { 0 };

  hex_encode ((const u8 *) luks->af_buf, luks->af_len, (u8 *) af_buf);

  // ct

  char ct_buf[LUKS_CT_HEX_LEN + 1] = { 0 };

  hex_encode ((const u8 *) luks->ct_buf, LUKS_CT_LEN, (u8 *) ct_buf);

  // output

  int line_len = snprintf (line_buf, line_size, "%s%s$%u$%u$%s$%s$%s",
    SIGNATURE_LUKS,
    cipher_mode,
    key_size,
    iterations,
    salt_buf,
    af_buf,
    ct_buf);

  return line_len;
}

void module_init (module_ctx_t *module_ctx)
{
  module_ctx->module_context_size             = MODULE_CONTEXT_SIZE_CURRENT;
  module_ctx->module_interface_version        = MODULE_INTERFACE_VERSION_CURRENT;

  module_ctx->module_attack_exec              = module_attack_exec;
  module_ctx->module_benchmark_esalt          = MODULE_DEFAULT;
  module_ctx->module_benchmark_hook_salt      = MODULE_DEFAULT;
  module_ctx->module_benchmark_mask           = MODULE_DEFAULT;
  module_ctx->module_benchmark_charset        = MODULE_DEFAULT;
  module_ctx->module_benchmark_salt           = MODULE_DEFAULT;
  module_ctx->module_bridge_name              = MODULE_DEFAULT;
  module_ctx->module_bridge_type              = MODULE_DEFAULT;
  module_ctx->module_build_plain_postprocess  = MODULE_DEFAULT;
  module_ctx->module_deep_comp_kernel         = MODULE_DEFAULT;
  module_ctx->module_deprecated_notice        = MODULE_DEFAULT;
  module_ctx->module_dgst_pos0                = module_dgst_pos0;
  module_ctx->module_dgst_pos1                = module_dgst_pos1;
  module_ctx->module_dgst_pos2                = module_dgst_pos2;
  module_ctx->module_dgst_pos3                = module_dgst_pos3;
  module_ctx->module_dgst_size                = module_dgst_size;
  module_ctx->module_dictstat_disable         = MODULE_DEFAULT;
  module_ctx->module_esalt_size               = module_esalt_size;
  module_ctx->module_extra_buffer_size        = MODULE_DEFAULT;
  module_ctx->module_extra_tmp_size           = MODULE_DEFAULT;
  module_ctx->module_extra_tuningdb_block     = MODULE_DEFAULT;
  module_ctx->module_forced_outfile_format    = MODULE_DEFAULT;
  module_ctx->module_hash_binary_count        = MODULE_DEFAULT;
  module_ctx->module_hash_binary_parse        = MODULE_DEFAULT;
  module_ctx->module_hash_binary_save         = MODULE_DEFAULT;
  module_ctx->module_hash_decode_postprocess  = MODULE_DEFAULT;
  module_ctx->module_hash_decode_potfile      = MODULE_DEFAULT;
  module_ctx->module_hash_decode_zero_hash    = MODULE_DEFAULT;
  module_ctx->module_hash_decode              = module_hash_decode;
  module_ctx->module_hash_encode_status       = MODULE_DEFAULT;
  module_ctx->module_hash_encode_potfile      = MODULE_DEFAULT;
  module_ctx->module_hash_encode              = module_hash_encode;
  module_ctx->module_hash_init_selftest       = MODULE_DEFAULT;
  module_ctx->module_hash_mode                = MODULE_DEFAULT;
  module_ctx->module_hash_category            = module_hash_category;
  module_ctx->module_hash_name                = module_hash_name;
  module_ctx->module_hashes_count_min         = MODULE_DEFAULT;
  module_ctx->module_hashes_count_max         = MODULE_DEFAULT;
  module_ctx->module_hlfmt_disable            = MODULE_DEFAULT;
  module_ctx->module_hook_extra_param_size    = MODULE_DEFAULT;
  module_ctx->module_hook_extra_param_init    = MODULE_DEFAULT;
  module_ctx->module_hook_extra_param_term    = MODULE_DEFAULT;
  module_ctx->module_hook12                   = MODULE_DEFAULT;
  module_ctx->module_hook23                   = MODULE_DEFAULT;
  module_ctx->module_hook_salt_size           = MODULE_DEFAULT;
  module_ctx->module_hook_size                = MODULE_DEFAULT;
  module_ctx->module_jit_build_options        = MODULE_DEFAULT;
  module_ctx->module_jit_cache_disable        = MODULE_DEFAULT;
  module_ctx->module_kernel_accel_max         = MODULE_DEFAULT;
  module_ctx->module_kernel_accel_min         = MODULE_DEFAULT;
  module_ctx->module_kernel_loops_max         = MODULE_DEFAULT;
  module_ctx->module_kernel_loops_min         = MODULE_DEFAULT;
  module_ctx->module_kernel_threads_max       = MODULE_DEFAULT;
  module_ctx->module_kernel_threads_min       = MODULE_DEFAULT;
  module_ctx->module_kern_type                = module_kern_type;
  module_ctx->module_kern_type_dynamic        = MODULE_DEFAULT;
  module_ctx->module_opti_type                = module_opti_type;
  module_ctx->module_opts_type                = module_opts_type;
  module_ctx->module_outfile_check_disable    = MODULE_DEFAULT;
  module_ctx->module_outfile_check_nocomp     = MODULE_DEFAULT;
  module_ctx->module_potfile_custom_check     = MODULE_DEFAULT;
  module_ctx->module_potfile_disable          = MODULE_DEFAULT;
  module_ctx->module_potfile_keep_all_hashes  = MODULE_DEFAULT;
  module_ctx->module_pwdump_column            = MODULE_DEFAULT;
  module_ctx->module_pw_max                   = MODULE_DEFAULT;
  module_ctx->module_pw_min                   = MODULE_DEFAULT;
  module_ctx->module_salt_max                 = MODULE_DEFAULT;
  module_ctx->module_salt_min                 = MODULE_DEFAULT;
  module_ctx->module_salt_type                = module_salt_type;
  module_ctx->module_separator                = MODULE_DEFAULT;
  module_ctx->module_st_hash                  = module_st_hash;
  module_ctx->module_st_pass                  = module_st_pass;
  module_ctx->module_tmp_size                 = module_tmp_size;
  module_ctx->module_unstable_warning         = module_unstable_warning;
  module_ctx->module_warmup_disable           = MODULE_DEFAULT;
}
